/**
 * Handles events from the page base.
 * 
 * @author Yoel Gluschnaider
 */
var eventsHandler = (function () 
{
	
	var me = {};
	
	// The event handlers registered.
	var m_eventsHandlers = {};
	
	// A running index which acts like the handle for each
	// event registration.
	var m_handle = 0;
	
	/**
	 * Called when the slaveMux receives an event messge.
	 * @param {String} eventName - the event's name
	 * @param {Object} data - the data of the event.
	 * 
	 * @author Yoel Gluschnaider
	 */
	me.eventReceived = function (eventName, data) 
	{
		var eventHandlers = m_eventsHandlers[eventName];
		// if there are events handlers registered on this event, dispatch them. 
		if (eventHandlers) 
		{
			for (var i in eventHandlers) 
			{
				eventHandlers[i](eventName, data);
			}
		}
	};
	
	/**
	 * Register on an event. 
	 *  
	 * @param {Object} eventName - the event's name.
	 * @param {Object} eventCb - the event's handler: eventCb(eventName).
	 * 
	 * @return {Object} eventHandle - the event handle to unregister.
	 * 
	 * @author Yoel Gluschnaider
	 */
	me.registerOnEvent = function (eventName, eventCb)
	{
	    if (!eventName || !eventCb)
        {
            return;
        }
		// save the current handle.
		var curHandle = m_handle;
		// if there is no event handlers on this event, create
		// the initial event handlers array.
		if (!m_eventsHandlers[eventName]) 
		{
			m_eventsHandlers[eventName] = {};
		}
		// push the event handler to the handlers array.
		m_eventsHandlers[eventName][curHandle] = eventCb;
		
		// increment the handle.
		m_handle++;
		
		// notify the master that we registered on an event.
		slaveDemux.registerOnEvent(eventName);
		
		return curHandle;
	};
	
	/**
	 * Unregisters from an event.
	 * @param {Object} eventName - the event name.
	 * @param {Object} eventHandle - the handle returned in the registration function.
	 */
	me.unregisterFromEvent = function (eventName, eventHandle) 
	{
		// delete the event handler CB.
		delete m_eventsHandlers[eventName][eventHandle];
		
		// notify the master that we unregistered from an event.
		slaveDemux.unregisterOnEvent(eventName);
	};
	
	return me;
	
})();
